<?php

namespace Mnv\Core\Uploads;

use Mnv\Core\Config;
use Mnv\Core\Filesystem\Filesystem;
use Mnv\Core\Filesystem\FilesystemFactory;
use Mnv\Core\Locale\I18N;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

class UploaderPdf extends Filesystem
{
    const IMAGETYPE_GIF = 1;
    const IMAGETYPE_JPEG = 2;
    const IMAGETYPE_PNG = 3;
    const IMAGETYPE_WEBP = 4;

    protected $config = [];
    protected $options = [];

    public bool $error = false;

    /** @var array  */
    public array $response = [];

    /** @var SymfonyUploadedFile|mixed  */
    protected SymfonyUploadedFile $file;

    /** @var ImageGenerator|ImagineGenerator */
    protected $generator;

    /** @var array  */
    public array $uploaded = [];
    public array $uploadable = [];

    public function __construct($options, $realPath, $path, $managerId)
    {

        $this->options = include __DIR__ . '/config/config.inc.php';
        if (!empty($options)) {
            $this->options = $options + $this->options;
        }

        $this->options['max_file_size'] = Config::getValue('max_file_size') * 1024 * 1024;
        $this->options['max_up_size']   = Config::getValue('max_up_size') * 1024 * 1024;

        $this->realPath = $realPath;
        $this->path     = $path;

        if (Config::getValue('image_generation') == 'imagine') {
            $this->generator = FilesystemFactory::imagineGenerator($this->realPath, $this->path, $managerId);
        } else {
            $this->generator = FilesystemFactory::imageGenerator($this->realPath, $this->path, $managerId);
        }

        // TODO: доработать `$this->request->file()`
        if (request()->hasFile('pdf')) {
            $files = request()->files->get('pdf', '');

            foreach ($files as $lang => $item) {
                $this->file = $item;

                $this->uploadable['lang_code'] = $lang;
                $this->uploadable['name']      = $this->file->getClientOriginalName();
                $this->uploadable['tmp_name']  = $this->file->getPathname();
                $this->uploadable['mimeType']  = $this->file->getClientMimeType();
                $this->uploadable['size']      = $this->file->getSize();
                $this->uploadable['extension'] = $this->file->getClientOriginalExtension();
                $this->uploadable['error']     = $this->file->getError();

                $this->validate();
            }
        }

    }

    /**
     * Валидация загружаемого файла
     * @return bool
     */
    public function validate(): bool
    {

        if (!empty($this->uploadable['error'])) {
            $this->error = true;
            $this->response = array('status' => 500, 'message' => $this->uploadable['error'], 'error' => $this->uploadable['error'], 'type' => 'error');
            return false;
        }

        $content_length = $this->fix_integer_overflow((int)$this->get_server_var('CONTENT_LENGTH'));
        $post_max_size = $this->get_config_bytes(\ini_get('post_max_size'));
        $upload_max_filesize = $this->get_config_bytes(\ini_get('upload_max_filesize'));

        if (($post_max_size && ($content_length > $post_max_size)) || ($upload_max_filesize && ($content_length > $upload_max_filesize))) {
            $this->error = true;
            $this->response = array('status' => 500, 'message' => lang('fileManager:errors:21'), 'error' => 'upload_max_filesize', 'type' => 'error');
            return false;
        }

        if (!empty($this->uploadable)) {

            if (!$this->file->getSize()) {
                $this->error = true;
                /** Загружаемый файл пустой, либо к нему невозможно получить доступ. */
                $this->response = array('status' => 500, 'message' => lang('fileManager:errors:13'), 'error' => 'file_size_empty', 'type' => 'error');
                return false;
            }
            /** Загруженный файл превышает параметр MAX_FILE_SIZE, указанный в HTML-форме */
            if ($this->options['max_file_size'] && $this->file->getSize() > ($this->options['max_file_size'])) {
                $this->error = true;
                /** Слишком большой размер файла */
                $this->response = array('status' => 500, 'message' => lang('fileManager:errors:26'), 'error' => 'max_file_size', 'type' => 'error');
                return false;
            }

            /** Файл слишком мал */
            if ($this->options['min_file_size'] && $this->file->getSize() < $this->options['min_file_size']) {
                $this->error = true;
                $this->response = array('status' => 500, 'message' => lang('fileManager:errors:25'), 'error' => 'min_file_size', 'type' => 'error');
                return false;
            }

            if ($this->path == '') {
                $this->error = true;
                /** Нельзя загружать файлы в корневую директорию */
                $this->response = array('status' => 500, 'message' => lang('fileManager:errors:18'), 'error' => 'root_path_error', 'type' => 'error');
                return false;
            }

            /** Если нет такой директории, то создаем */
            $this->ensureDirectoryExists($this->realPath);
            if ($this->exists($this->realPath . $this->uploadable['name'])) {
                $this->error = true;
                /** Файл с данным именем уже существует на сервере */
                $this->response = array('status' => 500, 'message' => lang('fileManager:errors:16'), 'error' => 'duplicate_file', 'type' => 'error');
                return false;
            }

            $this->upload();
        }

//        else {
//            /** Файл не удалось загрузить */
//            $this->response = array('status' => 500, 'message' => lang('fileManager:errors:27'), 'error' => 'no_file_was_uploaded', 'type' => 'error');
//            return false;
//        }

        return true;
    }

    /** Загрузка файла */
    public function upload(): void
    {
        $newFileName = substr(md5(number_format(time() * rand(),0,'','')),3,20);
        $this->uploadable['name'] = $newFileName . '.' . $this->uploadable['extension'];
        $this->file->move($this->realPath, $this->uploadable['name']);

        $uploadedId = $this->generator->saveFile($this->uploadable);
        if (!$uploadedId) {
            $this->error = true;
            /** Невозможно загрузить файл на сервер. */
            $this->response = array('status' => 500, 'message' => I18N::locale("Невозможно загрузить файл на сервер", "Rasm qo'shildi!", "Image added!"), 'error' => 'unable_upload_file', 'type' => 'error');
        }

        $this->uploaded[] = [
            'fileId'    => $uploadedId,
            'lang_code' => $this->uploadable['lang_code'],
        ];
        /** Файл загружен. */
        $this->response = array(
            'status'    => 200,
            'message'   => I18N::locale("Файл загружен.", "Rasm qo'shildi!", "Image added!"), 'error' => '',
            'fileId'    => $uploadedId,
            'lang_code' => $this->uploadable['lang_code'],
            'type'      => 'success'
        );
    }

    /**
     * Метод для получения картинки при загрузках в контент (tinymce)
     *
     * @param $fileId
     * @return string|null
     */
    public function getFileUrl($fileId): ?string
    {
        if ($file = connect('files')->where('fileId', $fileId)->get('array')) {
            $image = ImageSizes::init()->get(null, $file);
            return $image['original'] ?? '';
        }

        return null;
    }

    /**
     * Валидация изображения
     * @return bool
     */
    protected function is_valid_image_file(): bool
    {
        if (!preg_match('/\.(gif|jpe?g|png|webp)$/i', $this->file->getClientOriginalName())) {
            return false;
        }

        return !!$this->image_type();
    }

    protected function image_type()
    {
        $fp = fopen($this->file, 'r');
        $data = fread($fp, 4);
        fclose($fp);
        // GIF: 47 49 46 38
        if ($data === 'GIF8') {
            return self::IMAGETYPE_GIF;
        }
        // JPG: FF D8 FF
        if (bin2hex(substr($data, 0, 3)) === 'ffd8ff') {
            return self::IMAGETYPE_JPEG;
        }
        // PNG: 89 50 4E 47
        if (bin2hex(@$data[0]).substr($data, 1, 4) === '89PNG') {
            return self::IMAGETYPE_PNG;
        }
        if ($data === 'RIFF') {
            return self::IMAGETYPE_WEBP;
        }
        return false;
    }

    protected function get_image_size()
    {
        if ($this->options['image_library']) {

            if (extension_loaded('imagick')) {
                $image = new \Imagick();
                try {
                    if (@$image->pingImage($this->file)) {
                        $dimensions = array($image->getImageWidth(), $image->getImageHeight());
                        $image->destroy();

                        return $dimensions;
                    }

                    return false;

                } catch (\Exception $e) {
                    error_log($e->getMessage());
                }
            }

            if ($this->options['image_library'] === 2) {
                $cmd = $this->options['identify_bin'];
                $cmd .= ' -ping '.escapeshellarg($this->file);
                exec($cmd, $output, $error);
                if (!$error && !empty($output)) {
                    // image.jpg JPEG 1920x1080 1920x1080+0+0 8-bit sRGB 465KB 0.000u 0:00.000
                    $infos = preg_split('/\s+/', substr($output[0], strlen($this->file)));
                    return preg_split('/x/', $infos[2]);
                }

                return false;
            }
        }

        if (!function_exists('getimagesize')) {
            error_log('Function not found: getimagesize');

            return false;
        }

        return @getimagesize($this->file);
    }

    public function get_config_bytes($val): float
    {
        $val = trim($val);
        $last = strtolower($val[strlen($val)-1]);
        $val = (int)$val;
        switch ($last) {
            case 'g':
                $val *= 1024;
            case 'm':
                $val *= 1024;
            case 'k':
                $val *= 1024;
        }
        return $this->fix_integer_overflow($val);
    }

    // Исправлено переполнение 32-разрядных целых чисел со знаком,
    // работает для размеров до 2 ^ 32-1 байт (4 гигабайта - 1):
    protected function fix_integer_overflow($size): float
    {
        if ($size < 0) {
            $size += 2.0 * (PHP_INT_MAX + 1);
        }

        return $size;
    }

    /**
     * @param $id
     * @return false|mixed
     */
    protected function get_server_var($id)
    {
        if (isset($_SERVER[$id])) {
            return @$_SERVER[$id];
        }

        return false;
    }
}